/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.internal.transport.http;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.Writer;
import java.net.HttpCookie;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.text.MessageFormat;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.internal.storage.file.FileSnapshot;
import org.eclipse.jgit.internal.storage.file.LockFile;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.util.FileUtils;
import org.eclipse.jgit.util.IO;
import org.eclipse.jgit.util.RawParseUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class NetscapeCookieFile {
    private final Path path;
    private FileSnapshot snapshot;
    private byte[] hash;
    private final Instant createdAt;
    private Set<HttpCookie> cookies = null;
    private static final Logger LOG = LoggerFactory.getLogger(NetscapeCookieFile.class);

    public NetscapeCookieFile(Path path) {
        this(path, Instant.now());
    }

    NetscapeCookieFile(Path path, Instant createdAt) {
        this.path = path;
        this.snapshot = FileSnapshot.DIRTY;
        this.createdAt = createdAt;
    }

    public Path getPath() {
        return this.path;
    }

    public Set<HttpCookie> getCookies(boolean refresh) {
        block3: {
            if (this.cookies == null || refresh) {
                try {
                    byte[] in = this.getFileContentIfModified();
                    Set<HttpCookie> newCookies = NetscapeCookieFile.parseCookieFile(in, this.createdAt);
                    this.cookies = this.cookies != null ? NetscapeCookieFile.mergeCookies(newCookies, this.cookies) : newCookies;
                    return this.cookies;
                }
                catch (IOException | IllegalArgumentException e) {
                    LOG.warn(MessageFormat.format(JGitText.get().couldNotReadCookieFile, this.path), e);
                    if (this.cookies != null) break block3;
                    this.cookies = new LinkedHashSet<HttpCookie>();
                }
            }
        }
        return this.cookies;
    }

    private static Set<HttpCookie> parseCookieFile(byte[] input, Instant createdAt) throws IOException, IllegalArgumentException {
        String decoded = RawParseUtils.decode(StandardCharsets.US_ASCII, input);
        LinkedHashSet<HttpCookie> cookies = new LinkedHashSet<HttpCookie>();
        Throwable throwable = null;
        Object var5_6 = null;
        try (BufferedReader reader = new BufferedReader(new StringReader(decoded));){
            String line;
            while ((line = reader.readLine()) != null) {
                HttpCookie cookie = NetscapeCookieFile.parseLine(line, createdAt);
                if (cookie == null) continue;
                cookies.add(cookie);
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
        return cookies;
    }

    private static HttpCookie parseLine(String line, Instant createdAt) {
        long maxAge;
        if (line.isEmpty() || line.startsWith("#") && !line.startsWith("#HttpOnly_")) {
            return null;
        }
        String[] cookieLineParts = line.split("\t", 7);
        if (cookieLineParts == null) {
            throw new IllegalArgumentException(MessageFormat.format(JGitText.get().couldNotFindTabInLine, line));
        }
        if (cookieLineParts.length < 7) {
            throw new IllegalArgumentException(MessageFormat.format(JGitText.get().couldNotFindSixTabsInLine, cookieLineParts.length, line));
        }
        String name = cookieLineParts[5];
        String value = cookieLineParts[6];
        HttpCookie cookie = new HttpCookie(name, value);
        String domain = cookieLineParts[0];
        if (domain.startsWith("#HttpOnly_")) {
            cookie.setHttpOnly(true);
            domain = domain.substring("#HttpOnly_".length());
        }
        if (domain.startsWith(".")) {
            domain = domain.substring(1);
        }
        cookie.setDomain(domain);
        cookie.setPath(cookieLineParts[2]);
        cookie.setSecure(Boolean.parseBoolean(cookieLineParts[3]));
        long expires = Long.parseLong(cookieLineParts[4]);
        if (cookieLineParts[4].length() == 13) {
            expires = TimeUnit.MILLISECONDS.toSeconds(expires);
        }
        if ((maxAge = expires - createdAt.getEpochSecond()) <= 0L) {
            return null;
        }
        cookie.setMaxAge(maxAge);
        return cookie;
    }

    private byte[] getFileContentIfModified() throws IOException {
        int maxStaleRetries = 5;
        int retries = 0;
        File file = this.getPath().toFile();
        if (!file.exists()) {
            LOG.warn(MessageFormat.format(JGitText.get().missingCookieFile, file.getAbsolutePath()));
            return new byte[0];
        }
        while (true) {
            FileSnapshot oldSnapshot = this.snapshot;
            FileSnapshot newSnapshot = FileSnapshot.save(file);
            try {
                byte[] in = IO.readFully(file);
                byte[] newHash = NetscapeCookieFile.hash(in);
                if (Arrays.equals(this.hash, newHash)) {
                    if (oldSnapshot.equals(newSnapshot)) {
                        oldSnapshot.setClean(newSnapshot);
                    } else {
                        this.snapshot = newSnapshot;
                    }
                } else {
                    this.snapshot = newSnapshot;
                    this.hash = newHash;
                }
                return in;
            }
            catch (FileNotFoundException e) {
                throw e;
            }
            catch (IOException e) {
                if (FileUtils.isStaleFileHandle(e) && retries < 5) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug(MessageFormat.format(JGitText.get().configHandleIsStale, retries), e);
                    }
                    ++retries;
                    continue;
                }
                throw new IOException(MessageFormat.format(JGitText.get().cannotReadFile, this.getPath()), e);
            }
            break;
        }
    }

    private static byte[] hash(byte[] in) {
        return Constants.newMessageDigest().digest(in);
    }

    public void write(URL url) throws IOException, InterruptedException {
        Object cookiesFromFile;
        try {
            byte[] cookieFileContent = this.getFileContentIfModified();
            if (cookieFileContent != null) {
                LOG.debug("Reading the underlying cookie file '{}' as it has been modified since the last access", (Object)this.path);
                cookiesFromFile = NetscapeCookieFile.parseCookieFile(cookieFileContent, this.createdAt);
                this.cookies = NetscapeCookieFile.mergeCookies(cookiesFromFile, this.cookies);
            }
        }
        catch (FileNotFoundException cookieFileContent) {
            // empty catch block
        }
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        cookiesFromFile = null;
        Object var4_6 = null;
        try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)output, StandardCharsets.US_ASCII);){
            NetscapeCookieFile.write(writer, this.cookies, url, this.createdAt);
        }
        catch (Throwable throwable) {
            if (cookiesFromFile == null) {
                cookiesFromFile = throwable;
            } else if (cookiesFromFile != throwable) {
                ((Throwable)cookiesFromFile).addSuppressed(throwable);
            }
            throw cookiesFromFile;
        }
        LockFile lockFile = new LockFile(this.path.toFile());
        int retryCount = 0;
        while (retryCount < 4) {
            if (lockFile.lock()) {
                try {
                    lockFile.setNeedSnapshot(true);
                    lockFile.write(output.toByteArray());
                    if (!lockFile.commit()) {
                        throw new IOException(MessageFormat.format(JGitText.get().cannotCommitWriteTo, this.path));
                    }
                }
                finally {
                    lockFile.unlock();
                }
                return;
            }
            Thread.sleep(500L);
            ++retryCount;
        }
        throw new IOException(MessageFormat.format(JGitText.get().cannotLock, lockFile));
    }

    static void write(Writer writer, Collection<HttpCookie> cookies, URL url, Instant createdAt) throws IOException {
        for (HttpCookie cookie : cookies) {
            NetscapeCookieFile.writeCookie(writer, cookie, url, createdAt);
        }
    }

    private static void writeCookie(Writer writer, HttpCookie cookie, URL url, Instant createdAt) throws IOException {
        if (cookie.getMaxAge() <= 0L) {
            return;
        }
        String domain = "";
        if (cookie.isHttpOnly()) {
            domain = "#HttpOnly_";
        }
        domain = cookie.getDomain() != null ? String.valueOf(domain) + cookie.getDomain() : String.valueOf(domain) + url.getHost();
        writer.write(domain);
        writer.write("\t");
        writer.write("TRUE");
        writer.write("\t");
        String path = cookie.getPath();
        if (path == null) {
            path = url.getPath();
        }
        writer.write(path);
        writer.write("\t");
        writer.write(Boolean.toString(cookie.getSecure()).toUpperCase());
        writer.write("\t");
        String expirationDate = String.valueOf(createdAt.getEpochSecond() + cookie.getMaxAge());
        writer.write(expirationDate);
        writer.write("\t");
        writer.write(cookie.getName());
        writer.write("\t");
        writer.write(cookie.getValue());
        writer.write("\n");
    }

    static Set<HttpCookie> mergeCookies(Set<HttpCookie> cookies1, Set<HttpCookie> cookies2) {
        LinkedHashSet<HttpCookie> mergedCookies = new LinkedHashSet<HttpCookie>(cookies1);
        if (cookies2 != null) {
            mergedCookies.addAll(cookies2);
        }
        return mergedCookies;
    }
}

