/*
 * Decompiled with CFR 0.152.
 */
package io.netty.handler.codec.compression;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.ByteToByteDecoder;
import io.netty.handler.codec.compression.CompressionException;
import io.netty.handler.codec.compression.Snappy;
import io.netty.handler.codec.compression.SnappyChecksumUtil;
import java.nio.charset.Charset;
import java.util.Arrays;

public class SnappyFramedDecoder
extends ByteToByteDecoder {
    private static final byte[] SNAPPY = "sNaPpY".getBytes(Charset.forName("US-ASCII"));
    private final Snappy snappy = new Snappy();
    private final boolean validateChecksums;
    private boolean started;
    private boolean corrupted;

    public SnappyFramedDecoder() {
        this(false);
    }

    public SnappyFramedDecoder(boolean validateChecksums) {
        this.validateChecksums = validateChecksums;
    }

    @Override
    protected void decode(ChannelHandlerContext ctx, ByteBuf in, ByteBuf out) throws Exception {
        if (this.corrupted) {
            in.skipBytes(in.readableBytes());
            return;
        }
        try {
            int idx = in.readerIndex();
            int inSize = in.writerIndex() - idx;
            if (inSize < 4) {
                return;
            }
            short chunkTypeVal = in.getUnsignedByte(idx);
            ChunkType chunkType = SnappyFramedDecoder.mapChunkType((byte)chunkTypeVal);
            int chunkLength = in.getUnsignedByte(idx + 1) | in.getUnsignedByte(idx + 2) << 8 | in.getUnsignedByte(idx + 3) << 16;
            switch (chunkType) {
                case STREAM_IDENTIFIER: {
                    if (chunkLength != SNAPPY.length) {
                        throw new CompressionException("Unexpected length of stream identifier: " + chunkLength);
                    }
                    if (inSize < 4 + SNAPPY.length) break;
                    byte[] identifier = new byte[chunkLength];
                    in.skipBytes(4).readBytes(identifier);
                    if (!Arrays.equals(identifier, SNAPPY)) {
                        throw new CompressionException("Unexpected stream identifier contents. Mismatched snappy protocol version?");
                    }
                    this.started = true;
                    break;
                }
                case RESERVED_SKIPPABLE: {
                    if (!this.started) {
                        throw new CompressionException("Received RESERVED_SKIPPABLE tag before STREAM_IDENTIFIER");
                    }
                    if (inSize < 4 + chunkLength) {
                        return;
                    }
                    in.skipBytes(4 + chunkLength);
                    break;
                }
                case RESERVED_UNSKIPPABLE: {
                    throw new CompressionException("Found reserved unskippable chunk type: 0x" + Integer.toHexString(chunkTypeVal));
                }
                case UNCOMPRESSED_DATA: {
                    if (!this.started) {
                        throw new CompressionException("Received UNCOMPRESSED_DATA tag before STREAM_IDENTIFIER");
                    }
                    if (chunkLength > 65540) {
                        throw new CompressionException("Received UNCOMPRESSED_DATA larger than 65540 bytes");
                    }
                    if (inSize < 4 + chunkLength) {
                        return;
                    }
                    in.skipBytes(4);
                    if (this.validateChecksums) {
                        int checksum = in.readUnsignedByte() | in.readUnsignedByte() << 8 | in.readUnsignedByte() << 16 | in.readUnsignedByte() << 24;
                        ByteBuf data = in.readSlice(chunkLength - 4);
                        SnappyChecksumUtil.validateChecksum(data, checksum);
                        out.writeBytes(data);
                        break;
                    }
                    in.skipBytes(4);
                    in.readBytes(out, chunkLength - 4);
                    break;
                }
                case COMPRESSED_DATA: {
                    if (!this.started) {
                        throw new CompressionException("Received COMPRESSED_DATA tag before STREAM_IDENTIFIER");
                    }
                    if (inSize < 4 + chunkLength) {
                        return;
                    }
                    in.skipBytes(4);
                    int checksum = in.readUnsignedByte() | in.readUnsignedByte() << 8 | in.readUnsignedByte() << 16 | in.readUnsignedByte() << 24;
                    if (this.validateChecksums) {
                        ByteBuf uncompressed = ctx.alloc().buffer();
                        this.snappy.decode(in.readSlice(chunkLength - 4), uncompressed);
                        SnappyChecksumUtil.validateChecksum(uncompressed, checksum);
                        out.writeBytes(uncompressed);
                    } else {
                        this.snappy.decode(in.readSlice(chunkLength - 4), out);
                    }
                    this.snappy.reset();
                }
            }
        }
        catch (Exception e) {
            this.corrupted = true;
            throw e;
        }
    }

    static ChunkType mapChunkType(byte type) {
        if (type == 0) {
            return ChunkType.COMPRESSED_DATA;
        }
        if (type == 1) {
            return ChunkType.UNCOMPRESSED_DATA;
        }
        if (type == -128) {
            return ChunkType.STREAM_IDENTIFIER;
        }
        if ((type & 0x80) == 128) {
            return ChunkType.RESERVED_SKIPPABLE;
        }
        return ChunkType.RESERVED_UNSKIPPABLE;
    }

    static enum ChunkType {
        STREAM_IDENTIFIER,
        COMPRESSED_DATA,
        UNCOMPRESSED_DATA,
        RESERVED_UNSKIPPABLE,
        RESERVED_SKIPPABLE;

    }
}

